<?php
session_start();
require_once "config.php";

function parseSms($sender, $message) {
    $data = ['number' => 'N/A', 'amount' => 'N/A', 'trx_id' => 'N/A', 'datetime' => 'N/A'];
    if ($sender == 'bKash') {
        if (preg_match('/received Tk ([\d,]+\.\d{2}) from (\d{11})\. .* TrxID (\w+) at (.*)/', $message, $matches)) {
            $data['amount'] = $matches[1]; $data['number'] = $matches[2]; $data['trx_id'] = $matches[3]; $data['datetime'] = $matches[4];
        }
    } elseif ($sender == 'NAGAD') {
        if (preg_match('/Amount: Tk ([\d,]+\.\d{2})\s*Sender: (\d{11}).*TxnID: (\w+)\s*Balance:.*(\d{2}\/\d{2}\/\d{4} \d{2}:\d{2})/s', $message, $matches)) {
            $data['amount'] = $matches[1]; $data['number'] = $matches[2]; $data['trx_id'] = $matches[3]; $data['datetime'] = $matches[4];
        }
    }
    return $data;
}

$email = isset($_GET['email']) ? trim($_GET['email']) : '';
$password = isset($_GET['password']) ? trim($_GET['password']) : '';
$is_authenticated = false;
$sms_data = [];
$toast_message = null;

$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
if ($page < 1) $page = 1;
$limit = 8;
$offset = ($page - 1) * $limit;
$total_records = 0;
$total_pages = 0;

if (isset($_SESSION['toast'])) {
    $toast_message = $_SESSION['toast'];
    unset($_SESSION['toast']);
}

if (!$conn || $conn->connect_error) {
    $toast_message = ['type' => 'error', 'message' => 'Database connection failed.'];
} else {
    if (!empty($email) && !empty($password)) {
        $sql = "SELECT id FROM admin WHERE email = ? AND password = ?";
        if ($stmt = $conn->prepare($sql)) {
            $stmt->bind_param("ss", $email, $password);
            $stmt->execute();
            $stmt->store_result();
            if ($stmt->num_rows == 1) {
                $is_authenticated = true;
            }
            $stmt->close();
        }
    }

    if ($is_authenticated) {
        if ($_SERVER["REQUEST_METHOD"] == "POST") {
            $action_taken = true;
            if (isset($_POST['add_sms'])) {
                $sender = trim($_POST['sender']);
                $message = trim($_POST['message']);
                if (!empty($sender) && !empty($message)) {
                    $sql = "INSERT INTO rs_sms (sender, message, status) VALUES (?, ?, 'pending')";
                    if ($stmt = $conn->prepare($sql)) {
                        $stmt->bind_param("ss", $sender, $message);
                        if ($stmt->execute() && $stmt->affected_rows > 0) {
                            $_SESSION['toast'] = ['type' => 'success', 'message' => 'Record added successfully.'];
                        } else {
                            $_SESSION['toast'] = ['type' => 'error', 'message' => 'Error adding record.'];
                        }
                        $stmt->close();
                    }
                } else {
                    $_SESSION['toast'] = ['type' => 'error', 'message' => 'Sender and Message cannot be empty.'];
                }
            } elseif (isset($_POST['delete_sms'])) {
                $id = trim($_POST['id']);
                $sql = "DELETE FROM rs_sms WHERE id = ?";
                if ($stmt = $conn->prepare($sql)) {
                    $stmt->bind_param("i", $id);
                    if ($stmt->execute() && $stmt->affected_rows > 0) {
                        $_SESSION['toast'] = ['type' => 'success', 'message' => 'Record deleted successfully.'];
                    } else {
                        $_SESSION['toast'] = ['type' => 'error', 'message' => 'Error deleting record.'];
                    }
                    $stmt->close();
                }
            } elseif (isset($_POST['update_status'])) {
                $id = trim($_POST['id']);
                $status = trim($_POST['status']);
                $sql = "UPDATE rs_sms SET status = ? WHERE id = ?";
                if ($stmt = $conn->prepare($sql)) {
                    $stmt->bind_param("si", $status, $id);
                    if ($stmt->execute()) {
                        $_SESSION['toast'] = ['type' => 'success', 'message' => 'Status updated successfully.'];
                    } else {
                        $_SESSION['toast'] = ['type' => 'error', 'message' => 'Error updating status.'];
                    }
                    $stmt->close();
                }
            } else {
                $action_taken = false;
            }
            
            if ($action_taken) {
                header("Location: " . $_SERVER['REQUEST_URI']);
                exit;
            }
        }

        $search_param = "%" . $search . "%";
        
        $sql_count = "SELECT COUNT(*) FROM rs_sms WHERE message LIKE ?";
        if ($stmt_count = $conn->prepare($sql_count)) {
            $stmt_count->bind_param("s", $search_param);
            $stmt_count->execute();
            $stmt_count->bind_result($total_records);
            $stmt_count->fetch();
            $stmt_count->close();
        }
        $total_pages = ceil($total_records / $limit);
        if ($page > $total_pages && $total_records > 0) {
            $page = $total_pages;
            $offset = ($page - 1) * $limit;
        }

        $sql = "SELECT id, sender, message, status, received_at FROM rs_sms WHERE message LIKE ? ORDER BY id DESC LIMIT ? OFFSET ?";
        if ($stmt = $conn->prepare($sql)) {
            $stmt->bind_param("sii", $search_param, $limit, $offset);
            $stmt->execute();
            $result = $stmt->get_result();
            $sms_data = $result->fetch_all(MYSQLI_ASSOC);
            $stmt->close();
        }
    }
    $conn->close();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Transaction Dashboard</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: #0a021b;
            color: #d9d0f0;
            background-image: linear-gradient(rgba(255, 255, 255, 0.03) 1px, transparent 1px),
                              linear-gradient(90deg, rgba(255, 255, 255, 0.03) 1px, transparent 1px);
            background-size: 25px 25px;
            overflow-y: scroll;
        }
        .container-wrapper {
            max-width: 1500px;
            margin-left: auto;
            margin-right: auto;
            padding: 2rem 1.5rem;
        }
        .glass-card {
            background: rgba(22, 12, 59, 0.6);
            backdrop-filter: blur(12px);
            border: 1px solid rgba(138, 43, 226, 0.15);
            border-radius: 24px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.4);
        }
        .btn-premium {
            background-image: linear-gradient(to right, #7e22ce 0%, #a855f7 51%, #7e22ce 100%);
            background-size: 200% auto;
            color: white;
            padding: 12px 24px;
            border-radius: 12px;
            font-weight: 600;
            transition: all 0.5s cubic-bezier(0.25, 0.8, 0.25, 1);
            box-shadow: 0 6px 20px rgba(168, 85, 247, 0.3);
            border: none;
            cursor: pointer;
            display: inline-flex;
            align-items: center;
            gap: 0.6rem;
        }
        .btn-premium:hover {
            background-position: right center;
            box-shadow: 0 10px 30px rgba(168, 85, 247, 0.5);
            transform: translateY(-3px);
        }
        .btn-danger {
            background: rgba(239, 68, 68, 0.1);
            color: #f87171;
            border: 1px solid rgba(239, 68, 68, 0.3);
            border-radius: 50%;
            width: 38px;
            height: 38px;
            transition: all 0.3s;
            cursor: pointer;
            display: inline-flex;
            align-items: center;
            justify-content: center;
        }
        .btn-danger:hover {
            background: rgba(239, 68, 68, 0.2);
            color: #ef4444;
            box-shadow: 0 0 15px rgba(239, 68, 68, 0.3);
            transform: translateY(-2px) scale(1.05);
        }
        .input-field, .select-field {
            background-color: rgba(13, 5, 37, 0.7);
            border: 1px solid #4a2f8c;
            color: #e0d9ef;
            border-radius: 12px;
            padding: 12px;
            transition: all 0.3s;
        }
        .input-field:focus, .select-field:focus {
            outline: none;
            border-color: #a78bfa;
            box-shadow: 0 0 0 4px rgba(167, 139, 250, 0.2);
        }
        .select-field {
            -webkit-appearance: none;
            -moz-appearance: none;
            appearance: none;
            background-image: url('data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%22292.4%22%20height%3D%22292.4%22%3E%3Cpath%20fill%3D%22%23c4b5fd%22%20d%3D%22M287%2069.4a17.6%2017.6%200%200%200-13-5.4H18.4c-5%200-9.3%201.8-13%205.4A17.6%2017.6%200%200%200%200%2082.2c0%205%201.8%209.3%205.4%2013l128%20127.9c3.6%203.6%207.8%205.4%2013%205.4s9.4-1.8%2013-5.4L287%2095.2c3.6-3.6%205.4-7.8%205.4-13%200-5-1.8-9.3-5.4-13z%22%2F%3E%3C%2Fsvg%3E');
            background-repeat: no-repeat;
            background-position: right 12px center;
            background-size: 10px 10px;
        }
        th {
            background-color: rgba(35, 21, 79, 0.3);
            color: #c4b5fd;
            text-transform: uppercase;
            letter-spacing: 1px;
            font-weight: 600;
            padding: 18px 22px;
            text-align: left;
        }
        tr { border-bottom: 1px solid rgba(74, 47, 140, 0.3); }
        td { padding: 18px 22px; vertical-align: middle; }
        tbody tr { transition: background-color 0.3s, transform 0.3s; }
        tbody tr:hover {
            background-color: rgba(35, 21, 79, 0.7);
            transform: scale(1.01);
            box-shadow: 0 8px 30px rgba(0, 0, 0, 0.3);
            position: relative;
            z-index: 10;
        }
        tbody tr:last-child { border-bottom: none; }
        .table-wrapper {
            border: 1px solid rgba(138, 43, 226, 0.15);
            border-radius: 24px;
            overflow: hidden;
        }
        #toast {
            position: fixed;
            bottom: 25px;
            right: 25px;
            padding: 18px 24px;
            border-radius: 12px;
            background: rgba(22, 12, 59, 0.85);
            backdrop-filter: blur(10px);
            border: 1px solid rgba(138, 43, 226, 0.2);
            color: white;
            display: flex;
            align-items: center;
            gap: 12px;
            z-index: 9999;
            opacity: 0;
            transform: translateY(20px);
            transition: opacity 0.5s, transform 0.5s;
        }
        #toast.toast-glow-success {
            box-shadow: 0 5px 20px rgba(0, 0, 0, 0.3), 0 0 15px rgba(16, 185, 129, 0.3);
        }
        #toast.toast-glow-error {
            box-shadow: 0 5px 20px rgba(0, 0, 0, 0.3), 0 0 15px rgba(239, 68, 68, 0.3);
        }
        #toast.show { opacity: 1; transform: translateY(0); }
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        .fade-in { animation: fadeIn 0.6s ease-out; }
        .fade-in-delay { animation: fadeIn 0.6s ease-out 0.2s; animation-fill-mode: backwards; }
        
        .modal-overlay {
            position: fixed;
            inset: 0;
            background: rgba(10, 2, 27, 0.8);
            backdrop-filter: blur(5px);
            z-index: 50;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0;
            transition: opacity 0.3s ease;
            pointer-events: none;
        }
        .modal-overlay:not(.hidden) {
            opacity: 1;
            pointer-events: auto;
        }
        .modal-content {
            transform: scale(0.95);
            transition: transform 0.3s ease;
        }
        .modal-overlay:not(.hidden) .modal-content {
            transform: scale(1);
        }

        .pagination-link {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            width: 40px;
            height: 40px;
            border-radius: 10px;
            font-weight: 500;
            color: #c4b5fd;
            transition: all 0.3s;
            border: 1px solid transparent;
        }
        .pagination-link:hover {
            background-color: rgba(74, 47, 140, 0.5);
            color: #fff;
        }
        .pagination-link.active {
            background-color: #7e22ce;
            border-color: #a855f7;
            color: #fff;
            box-shadow: 0 4px 15px rgba(126, 34, 206, 0.3);
        }
        .pagination-link.disabled {
            color: #5a4a8c;
            pointer-events: none;
        }

    </style>
</head>
<body class="min-h-screen">

    <?php if ($toast_message): ?>
        <div id="toast" class="<?php echo $toast_message['type'] == 'success' ? 'toast-glow-success' : 'toast-glow-error'; ?>">
            <i class="fa-solid <?php echo $toast_message['type'] == 'success' ? 'fa-check-circle text-green-400 text-xl' : 'fa-times-circle text-red-400 text-xl'; ?>"></i>
            <span class="font-medium"><?php echo htmlspecialchars($toast_message['message']); ?></span>
        </div>
    <?php endif; ?>

    <div class="modal-overlay hidden" id="modal-overlay">
        <div class="glass-card modal-content w-full max-w-2xl">
            <form method="post" action="<?php echo htmlspecialchars($_SERVER["REQUEST_URI"]); ?>">
                <input type="hidden" name="add_sms" value="1">
                <div class="flex justify-between items-center p-6 border-b border-b-[rgba(138,43,226,0.15)]">
                    <h2 class="text-2xl font-semibold text-white">
                        <i class="fa-solid fa-plus-circle mr-2 text-purple-400"></i>
                        Add New Record
                    </h2>
                    <button type="button" class="text-purple-300 hover:text-white" id="close-modal-btn">
                        <i class="fa-solid fa-times fa-xl"></i>
                    </button>
                </div>
                <div class="p-8 grid grid-cols-1 gap-6">
                    <div>
                        <label for="modal-sender" class="block text-sm font-medium text-purple-300 mb-2">Sender</label>
                        <select name="sender" id="modal-sender" class="w-full select-field">
                            <option value="bKash">bKash</option>
                            <option value="NAGAD">NAGAD</option>
                            <option value="Other">Other</option>
                        </select>
                    </div>
                    <div>
                        <label for="modal-message" class="block text-sm font-medium text-purple-300 mb-2">Message</label>
                        <textarea name="message" id="modal-message" rows="4" class="w-full input-field" placeholder="SMS message content..."></textarea>
                    </div>
                </div>
                <div class="p-6 bg-[rgba(13,5,37,0.7)] rounded-b-2xl text-right">
                    <button type="submit" class="btn-premium w-full sm:w-auto justify-center">
                        <i class="fa-solid fa-check"></i>
                        Submit Record
                    </button>
                </div>
            </form>
        </div>
    </div>

    <div class="container-wrapper">
        <header class="flex flex-col sm:flex-row justify-between sm:items-center mb-8 fade-in">
            <h1 class="text-4xl font-bold text-white mb-4 sm:mb-0">
                <i class="fa-solid fa-file-invoice-dollar mr-3 text-purple-400"></i>
                Transaction Dashboard
            </h1>
        </header>

        <?php if ($is_authenticated): ?>
            <div class="mb-8 flex flex-col md:flex-row justify-between md:items-center gap-4 fade-in">
                <form method="GET" action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>" class="relative w-full md:max-w-md">
                    <input type="hidden" name="email" value="<?php echo htmlspecialchars($email); ?>">
                    <input type="hidden" name="password" value="<?php echo htmlspecialchars($password); ?>">
                    <input type="text" name="search" class="input-field w-full !pl-12" placeholder="Search by TrxID or Number..." value="<?php echo htmlspecialchars($search); ?>">
                    <i class="fa-solid fa-search text-purple-300 absolute left-4 top-1/2 -translate-y-1/2"></i>
                    <button type="submit" class="absolute right-2 top-1/2 -translate-y-1/2 btn-premium !py-2 !px-4 !rounded-lg">
                        Search
                    </button>
                </form>
                <button type="button" class="btn-premium w-full md:w-auto justify-center" id="add-btn">
                    <i class="fa-solid fa-plus"></i>
                    Add Transaction
                </button>
            </div>

            <div class="glass-card fade-in-delay table-wrapper">
                <div class="overflow-x-auto">
                    <table class="w-full min-w-[1200px] text-sm">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Sender</th>
                                <th>Number</th>
                                <th>Amount</th>
                                <th>TrxID</th>
                                <th>Date & Time</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (!empty($sms_data)) : ?>
                                <?php foreach ($sms_data as $row) : ?>
                                    <?php $parsed_data = parseSms($row['sender'], $row['message']); ?>
                                    <tr>
                                        <td class="font-medium text-purple-300"><?php echo htmlspecialchars($row['id']); ?></td>
                                        <td>
                                            <span class="font-bold <?php echo $row['sender'] == 'bKash' ? 'text-pink-400' : 'text-orange-400'; ?>">
                                                <i class="fa-solid <?php echo $row['sender'] == 'bKash' ? 'fa-mobile-screen-button' : 'fa-wallet'; ?> mr-2"></i>
                                                <?php echo htmlspecialchars($row['sender']); ?>
                                            </span>
                                        </td>
                                        <td><?php echo htmlspecialchars($parsed_data['number']); ?></td>
                                        <td class="font-medium text-green-400">Tk <?php echo htmlspecialchars($parsed_data['amount']); ?></td>
                                        <td class="font-mono text-purple-300"><?php echo htmlspecialchars($parsed_data['trx_id']); ?></td>
                                        <td><?php echo htmlspecialchars($parsed_data['datetime']); ?></td>
                                        <td>
                                            <form method="post" action="<?php echo htmlspecialchars($_SERVER["REQUEST_URI"]); ?>">
                                                <input type="hidden" name="update_status" value="1">
                                                <input type="hidden" name="id" value="<?php echo $row['id']; ?>">
                                                <select name="status" class="select-field !text-xs !py-1" onchange="this.form.submit()">
                                                    <option value="success" <?php echo $row['status'] == 'success' ? 'selected' : ''; ?>>Success</option>
                                                    <option value="pending" <?php echo $row['status'] == 'pending' ? 'selected' : ''; ?>>Pending</option>
                                                    <option value="failed" <?php echo $row['status'] == 'failed' ? 'selected' : ''; ?>>Failed</option>
                                                </select>
                                            </form>
                                        </td>
                                        <td class="text-center">
                                            <form method="post" action="<?php echo htmlspecialchars($_SERVER["REQUEST_URI"]); ?>" onsubmit="return confirm('Are you sure you want to delete this record?');">
                                                <input type="hidden" name="delete_sms" value="1">
                                                <input type="hidden" name="id" value="<?php echo $row['id']; ?>">
                                                <button type="submit" class="btn-danger">
                                                    <i class="fa-solid fa-trash-can"></i>
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php else : ?>
                                <tr>
                                    <td colspan="8" class="text-center p-12 text-purple-300">
                                        <i class="fa-solid fa-database fa-3x mb-4"></i>
                                        <p class="text-lg"><?php echo !empty($search) ? 'No records match your search.' : 'No data found.'; ?></p>
                                    </td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <?php if ($total_pages > 1) : ?>
                <nav class="flex justify-center mt-8 space-x-2 fade-in">
                    <?php
                        $base_url = "?email=" . urlencode($email) . "&password=" . urlencode($password) . "&search=" . urlencode($search);
                        $is_prev_disabled = ($page <= 1);
                        $is_next_disabled = ($page >= $total_pages);
                    ?>
                    <a href="<?php echo $is_prev_disabled ? '#' : $base_url . '&page=' . ($page - 1); ?>" class="pagination-link <?php echo $is_prev_disabled ? 'disabled' : ''; ?>">
                        <i class="fa-solid fa-arrow-left"></i>
                    </a>
                    
                    <?php for ($i = 1; $i <= $total_pages; $i++) : ?>
                        <a href="<?php echo $base_url . '&page=' . $i; ?>" class="pagination-link <?php echo $i == $page ? 'active' : ''; ?>">
                            <?php echo $i; ?>
                        </a>
                    <?php endfor; ?>

                    <a href="<?php echo $is_next_disabled ? '#' : $base_url . '&page=' . ($page + 1); ?>" class="pagination-link <?php echo $is_next_disabled ? 'disabled' : ''; ?>">
                        <i class="fa-solid fa-arrow-right"></i>
                    </a>
                </nav>
            <?php endif; ?>

        <?php else: ?>
             <div class="glass-card p-16 text-center fade-in">
                <i class="fa-solid fa-lock fa-4x text-red-400 mb-6 animate-pulse"></i>
                <h2 class="text-3xl font-bold text-white">Authentication Required</h2>
            </div>
        <?php endif; ?>

    </div>
    
    <script>
        document.addEventListener('DOMContentLoaded', (event) => {
            const toast = document.getElementById('toast');
            if (toast) {
                setTimeout(() => {
                    toast.classList.add('show');
                }, 100);
                setTimeout(() => {
                    toast.classList.remove('show');
                }, 3000);
            }

            const addBtn = document.getElementById('add-btn');
            const modalOverlay = document.getElementById('modal-overlay');
            const closeModalBtn = document.getElementById('close-modal-btn');

            if (addBtn) {
                addBtn.addEventListener('click', () => {
                    modalOverlay.classList.remove('hidden');
                });
            }

            if (closeModalBtn) {
                closeModalBtn.addEventListener('click', () => {
                    modalOverlay.classList.add('hidden');
                });
            }

            if (modalOverlay) {
                modalOverlay.addEventListener('click', (e) => {
                    if (e.target === modalOverlay) {
                        modalOverlay.classList.add('hidden');
                    }
                });
            }
        });
    </script>

</body>
</html>